package gov.va.vinci.dart.rule;

import gov.va.vinci.dart.biz.DartRequest;
import gov.va.vinci.dart.biz.Request;
import gov.va.vinci.dart.biz.DataSource;
import gov.va.vinci.dart.biz.DocumentTemplate;
import gov.va.vinci.dart.biz.Group;
import gov.va.vinci.dart.biz.Location;
import gov.va.vinci.dart.biz.Participant;

/**
 * Class CapriDocumentRuleEvaluator.
 *
 */
public class CapriDocumentRuleEvaluator extends AbstractDocumentRuleEvaluator {

    private static final String EHR_ACCESS_REQUEST_FORM = "EHR Access Request Form";

    /**
     * Is a specific document required for a site?
     * 
     */
    @Override
    public boolean isRequired(final Request incomingRequest, final DataSource dataSources,
            final DocumentTemplate documentTemplate, final Location location, final boolean isIndependentOnly) {

        if (incomingRequest == null) {
            return false;
        }

        Location primarySite = incomingRequest.getPrimaryLocation();
        if (isDartRequest(incomingRequest)) {

            DartRequest dartRequest = (DartRequest) incomingRequest;

            // Defect #134 - all of these documents are required once per research study.
            
            if (primarySite != null && location.getId() == primarySite.getId()) {

                // Research Request Memo is required for the primary site
                if (documentTemplate.getName().startsWith(RESEARCH_REQUEST_MEMO)) {
                    return true;
                }

                // Research Protocol is required for the primary site
                if (documentTemplate.getName().startsWith(RESEARCH_PROTOCOL)) {
                    return true;
                }

                // Sample Informed Consent is required for the primary site if the HIPAA patient consent is true
                if (documentTemplate.getName().startsWith(SAMPLE_INFORMED_CONSENT) && dartRequest.isHipaaConsent()) {
                    return true;
                }

                // As per DUA document item 1.10, include DUA if transferredexternal = true
                // Data Use Agreement is required for the primary site
                if (documentTemplate.getName().startsWith(DATA_USE_AGREEMENT) && dartRequest.isTransferredExternal()) {
                    return true;
                }

                // As of 8/13/2014 (Real SSN BRD item 1.2):
                // Real SSN Access Request is required for the primary site, if realSSN is selected
                if (documentTemplate.getName().startsWith(REAL_SSN_ACCESS_REQUEST)
                        && dartRequest.isRealSSN()) {

                    // As of 2/25/2015: Real SSN Access Request no longer required for any Independent Approver Workflow
                    // if( dartRequest.getWorkflowTypeId() == WorkflowResolver.WF_NDS ) {
                    if (!isIndependentOnly) {
                        return true;
                    }
                }
            }

            // As of 9/19/2014, CAPRI: Research Study Institutional Review Board (IRB) Approval is required of all sites (NOT
            // just
            // for primary site)
            // Research Study Institutional Review Board (IRB) Approval is required of all sites
            if (documentTemplate.getName().startsWith(RESEARCH_STUDY_INSTITUTIONAL_REVIEW_BOARD_IRB_APPROVAL)) {
                return true;
            }

            // As of 9/19/2014, CAPRI: Research and Development (RD) Committee Approval is required of all sites (NOT just for
            // primary site)
            // Research and Development (RD) Committee Approval is required of all sites
            if (documentTemplate.getName().startsWith(RESEARCH_AND_DEVELOPMENT_RD_COMMITTEE_APPROVAL)) {
                return true;
            }

            // As of 9/19/2014, CAPRI: Waiver of HIPAA Authorization is required of all sites if the HIPAA waiver is true (NOT
            // just
            // for primary site)
            // IRB Approval of Waiver of HIPAA is required of all sites if the HIPAA waiver is true
            if (documentTemplate.getName().startsWith(IRB_APPROVAL_OF_WAIVER_OF_HIPAA) && dartRequest.isHipaaWaiver()) {
                return true;
            }
        } else if (isPreparatoryRequest(incomingRequest)){
            if (primarySite != null && location.getId() == primarySite.getId()) {

                // Research Request Memo is required for the primary site
                if (documentTemplate.getName().startsWith(PREPARATORY_TO_RESEARCH_MEMORANDUM)) {
                    return true;
                }
            }
            
        }

        return false;
    }

    @Override
    public boolean isRequired(final Request request, final DataSource dataSource, final DocumentTemplate documentTemplate,
            final Participant participant, final boolean isIndependentOnly) {

        // Each participant in a study with CAPRI data should be required to upload an "EHR Access Request Form"
    //    if (participant.getPrincipalInvestigator() || participant.getDataAccess().booleanValue()) {
          if (participant.getCapriAccess().booleanValue()) {
            if (documentTemplate.getName().startsWith(EHR_ACCESS_REQUEST_FORM)) {
                return true;
            }
        }

        return false;
    }

    @Override
    public boolean isRequired(final Request req, final DataSource ds, final DocumentTemplate temp, final Group admin,
            final Location site, final boolean isIndependentOnly) {
        return false;
    }

    @Override
    public boolean isRequired(final Request req, final DataSource ds, final DocumentTemplate temp, final Group admin,
            final Participant participant, final boolean isIndependentOnly) {
        return false;
    }
}
